#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN_NEW = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmLogInSupplies;
using BROKEN_EDIT = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmRecordSupplyUsage;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{

	
	


	[TestFixture]
	public class BOL_Supply : BaseTest 
	{

		private BOL.Supply _tstSupply;
		private DataTable _dtSupply;
		

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				this._dtSupply = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Supply WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);

				this.RefreshData = false;
			}

			this._tstSupply = new BOL.Supply(this._dtSupply.Rows[0]);
			this._tstSupply.SetInitialBrokenRules(BROKEN_NEW.FormClassName, BROKEN_NEW.RuleSets.NewSupply);
		}

		[Test]
		public void Ctor_Guid_Pass()
		{
			System.Guid unitGuid = this._tstSupply.SupplyGuid;

			BOL.Supply tmpSupply = new BOL.Supply(unitGuid);

			Assert.AreEqual(unitGuid, tmpSupply.SupplyGuid, TABLES.Supply.SupplyGuid);
			Assert.AreEqual(this._tstSupply.SupplyName, tmpSupply.SupplyName, TABLES.Supply.SupplyName);
		}

		[Test]
		public void Ctor_Guid_Fail()
		{
			System.Guid unitGuid = System.Guid.NewGuid();

			BOL.Supply tmpSupply = new BOL.Supply(unitGuid);

			Assert.AreEqual(unitGuid, tmpSupply.SupplyGuid, TABLES.Supply.SupplyGuid);
			Assert.AreEqual(true, tmpSupply.IsNew, "IsNew");
		}


		[Test]
		public void Ctor_DataRow_Pass()
		{
			BOL.Supply tmpSupply = new BOL.Supply(this._dtSupply.Rows[0]);

			Assert.AreEqual(this._tstSupply.SupplyGuid, tmpSupply.SupplyGuid, TABLES.Supply.SupplyGuid);
			Assert.AreEqual(this._tstSupply.SupplyName, tmpSupply.SupplyName, TABLES.Supply.SupplyName);
		}

		[Test]
		public void Ctor_DataRow_Fail()
		{
			DataTable dtTest = this._dtSupply.Clone();

			//TODO: The LoadFromDataRow method in Supply should check for null on these fields... it doesn't and could crash in app
			dtTest.Columns.Remove(TABLES.Supply.LastUpdateDate);
			dtTest.Columns.Remove(TABLES.Supply.LastUpdateUser);
			dtTest.Columns.Remove(TABLES.Supply.RowVersion);
			
			BOL.Supply tmpSupply = new BOL.Supply(dtTest.NewRow());

			Assert.AreEqual(System.Guid.Empty, tmpSupply.SupplyGuid, TABLES.Supply.SupplyGuid);
			Assert.AreEqual(string.Empty, tmpSupply.SupplyName, TABLES.Supply.SupplyName);
		}

		[Test]
		public void AvailableQuantity_Pass()
		{
			this._tstSupply.LotQuantity = 999;		//To avoid the broken rule

			int tmpValue = new Random().Next(998);

			this._tstSupply.AvailableQuantity = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.AvailableQuantity, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_EDIT.AvailableQuantity), "BrokenRule");
		}

		[Test]
		public void AvailableQuantity_Fail()
		{
			this._tstSupply.SetInitialBrokenRules(BROKEN_EDIT.FormClassName, BROKEN_EDIT.RuleSets.UpdateSupply);

			this._tstSupply.AvailableQuantity = -1;

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_EDIT.AvailableQuantity), "BrokenRule");
		}

		[Test]
		public void SupplyName_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstSupply.SupplyName = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.SupplyName, "SupplyName");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.SupplyName), "BrokenRule");
		}

		[Test]
		public void SupplyName_Fail()
		{
			this._tstSupply.SupplyName = string.Empty;

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.SupplyName), "BrokenRule");
		}

		[Test]
		public void ReceiptDate_Pass()
		{
			System.DateTime tmpValue = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			this._tstSupply.ReceiptDate = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.ReceiptDate, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.ReceiptDate), "BrokenRule");
		}

		[Test]
		public void ReceiptDate_Fail()
		{
			this._tstSupply.ReceiptDate = System.DateTime.Now.AddYears(1);
			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.ReceiptDate), "BrokenRule-Future");

			//Validate the LotExpiration Rule breaking
			this._tstSupply.ReceiptDate = System.DateTime.Now.AddDays(-4);
			this._tstSupply.LotExpirationDate = System.DateTime.Now.AddDays(-3);
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotExpirationDate), "BrokenRule LotExpiration is ok");

			this._tstSupply.ReceiptDate = System.DateTime.Now.AddDays(-2);
			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotExpirationDate), "BrokenRule >LotExpiration");
		}

		[Test]
		public void ReceiptTechID_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstSupply.ReceiptTechId = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.ReceiptTechId);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ReceiptTechID_Fail()
		{
		}

		[Test]
		public void InvoiceNumber_Pass()
		{
			string tmpValue = new Random().Next(1111111,9999999).ToString();

			this._tstSupply.InvoiceNumber = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.InvoiceNumber, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.InvoiceNumber), "BrokenRule");
		}

		[Test]
		public void InvoiceNumber_Fail()
		{
			this._tstSupply.InvoiceNumber = string.Empty;

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.InvoiceNumber), "BrokenRule");
		}

		[Test]
		public void LotQuantity_Pass()
		{
			int tmpValue = new Random().Next(999);

			this._tstSupply.LotQuantity = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.LotQuantity, "Value" );
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotQuantity), "BrokenRule");
		}

		[Test]
		public void LotQuantity_Fail()
		{
			this._tstSupply.LotQuantity = 1000;
			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotQuantity), "BrokenRule");
		}

		[Test]
		public void InventoryChangeDate_Pass()
		{
			System.DateTime tmpValue = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			this._tstSupply.InventoryChangeDate = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.InventoryChangeDate, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_EDIT.InventoryChangeDate), "BrokenRule");
		}

		[Test]
		public void InventoryChangeDate_Fail()
		{
			this._tstSupply.SetInitialBrokenRules(BROKEN_EDIT.FormClassName, BROKEN_EDIT.RuleSets.UpdateSupply);

			this._tstSupply.InventoryChangeDate = System.DateTime.Now.AddYears(1);

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_EDIT.InventoryChangeDate), "BrokenRule");
		}

		[Test]
		public void LotExpirationDate_Pass()
		{
			System.DateTime tmpValue = System.DateTime.Now.AddYears(1);

			this._tstSupply.ReceiptDate = System.DateTime.Now;		//To avoid broken rule
			this._tstSupply.LotExpirationDate = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.LotExpirationDate, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotExpirationDate), "BrokenRule");
		}

		[Test]
		public void LotExpirationDate_Fail()
		{
			this._tstSupply.LotExpirationDate = System.DateTime.MinValue;
			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotExpirationDate), "BrokenRule");
		}

		[Test]
		public void SatisfactoryIndicator_Pass()
		{
			bool tmpValue = (!this._tstSupply.SatisfactoryIndicator);

			this._tstSupply.Comments = System.DateTime.Now.ToString();
			this._tstSupply.SatisfactoryIndicator = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.SatisfactoryIndicator, "Value" );
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.Comments), "BrokenRule");
		}

		[Test]
		public void SatisfactoryIndicator_Fail()
		{
			this._tstSupply.IsNew = true;
			this._tstSupply.Comments = string.Empty;
			this._tstSupply.SatisfactoryIndicator = false;

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.Comments), "BrokenRule");
		}

		[Test]
		public void Comments_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstSupply.Comments = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.Comments, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.Comments), "BrokenRule");
		}

		[Test]
		public void Comments_Fail()
		{
			this._tstSupply.SatisfactoryIndicator = true;
			this._tstSupply.Comments = "".PadRight(256,'X');

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.Comments), "BrokenRule");
		}

		[Test]
		public void SupplyGuid_Pass()
		{
			System.Guid tmpValue = System.Guid.NewGuid();
			
			BOL.Supply tmpSupply = new BOL.Supply(tmpValue);

			Assert.AreEqual(tmpValue, tmpSupply.SupplyGuid);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SupplyGuid_Fail()
		{
		}

		[Test]
		public void LotNumber_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstSupply.LotNumber = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.LotNumber, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotNumber), "BrokenRule");
		}

		[Test]
		public void LotNumber_Fail()
		{
			this._tstSupply.LotNumber = string.Empty;

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.LotNumber), "BrokenRule");
		}

		[Test]
		public void SupplyTypeId_Pass()
		{
			int tmpValue = new Random().Next(1,14);

			this._tstSupply.SupplyTypeId = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.SupplyTypeId, "Value");
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.SupplyType), "BrokenRule");
		}

		[Test]
		public void SupplyTypeId_Fail()
		{
			this._tstSupply.SupplyTypeId = 0;

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.SupplyType), "BrokenRule");
		}

		[Test]
		public void SupplyTypeText_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstSupply.SupplyTypeText= tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.SupplyTypeText);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SupplyTypeText_Fail()
		{
		}

		[Test]
		public void ManufacturerName_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();

			this._tstSupply.ManufacturerName = tmpValue;

			Assert.AreEqual(tmpValue, this._tstSupply.ManufacturerName);
			Assert.AreEqual(false, this._tstSupply.IsRuleBroken(BROKEN_NEW.ManufacturerName), "BrokenRule");
		}

		[Test]
		public void ManufacturerName_Fail()
		{
			this._tstSupply.ManufacturerName = string.Empty;

			Assert.AreEqual(true, this._tstSupply.IsRuleBroken(BROKEN_NEW.ManufacturerName), "BrokenRule");
		}

		[Test]
		public void DivisionCode_Pass()
		{
			int tmpValue = new Random().Next(99999);

			this._tstSupply.DivisionCode = tmpValue.ToString();

			Assert.AreEqual(tmpValue.ToString(), this._tstSupply.DivisionCode);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void DivisionCode_Fail()
		{
			this._tstSupply.DivisionCode = string.Empty;
		}

		[Test]
		public void LotException_Pass()
		{
			BOL.ExceptionReport tstValue = new BOL.ExceptionReport();

			this._tstSupply.LotException = tstValue;

			Assert.AreEqual(tstValue, this._tstSupply.LotException);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LotException_Fail()
		{
		}

		[Test]
		public void GetSupply_Pass()
		{
			DataTable dtTest = this._tstSupply.GetSupply();

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetSupply_Fail()
		{
		}

		[Test]
		public void GetSupplyList_Pass()
		{
			//No invoice number supplied
			System.Collections.ArrayList tmpList = BOL.Supply.GetSupplyList(Common.SupplyType.AllSupplies, false, string.Empty);
			Assert.AreEqual(true, tmpList.Count >= 0, "Count-Blank Invoice filter");

			//Invoice filter parameter
			System.Collections.ArrayList tmpList2 = BOL.Supply.GetSupplyList(Common.SupplyType.AllSupplies, false, this._tstSupply.InvoiceNumber);
			Assert.AreEqual(true, tmpList2.Count >= 0, "Count-Invoice Filter");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetSupplyList_Fail()
		{
		}
		
		[Test]
		public void GetSupplyByInvoiceLotNumber_Pass()
		{
			System.Guid unitGuid = this._tstSupply.GetSupplyByInvoiceLotNumber();

			Assert.AreEqual(unitGuid, this._tstSupply.SupplyGuid);
		}

		[Test]
		public void GetSupplyByInvoiceLotNumber_Fail()
		{
			this._tstSupply.LotNumber = "Pumpkin";

			System.Guid unitGuid = this._tstSupply.GetSupplyByInvoiceLotNumber();

			Assert.AreEqual(System.Guid.Empty, unitGuid);
		}

		
		[Test]
		public void GetSupplyDetails_Pass()
		{
			System.Collections.ArrayList alTest = BOL.Supply.GetSupplyDetails();

			Assert.AreEqual(true, alTest.Count >= 0, "Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetSupplyDetails_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			System.DateTime tstValue = System.DateTime.Now.AddDays(-1);

			//For code coverage----
			this._tstSupply.Comments = tstValue.ToString();
			this._tstSupply.InventoryChangeDate = tstValue;

			DataRow dr = this._tstSupply.LoadDataRowFromThis(this._dtSupply.NewRow());

			Assert.AreEqual(this._tstSupply.SupplyName, dr[TABLES.Supply.SupplyName].ToString(), TABLES.Supply.SupplyName);
			Assert.AreEqual(this._tstSupply.SupplyTypeId.ToString(), dr[TABLES.SupplyType.SupplyTypeId].ToString(), TABLES.SupplyType.SupplyTypeId);
			Assert.AreEqual(this._tstSupply.ManufacturerName, dr[TABLES.Supply.ManufacturerName].ToString(), TABLES.Supply.ManufacturerName);

			//Manual sort -- 
			Assert.AreEqual(this._tstSupply.Comments, dr[TABLES.Supply.Comments].ToString(), TABLES.Supply.Comments);
			Assert.AreEqual(this._tstSupply.InventoryChangeDate , (System.DateTime)dr[TABLES.Supply.InventoryChangeDate], TABLES.Supply.InventoryChangeDate);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}

		[Test]
		public void GetEmptySupplyTable_Pass()
		{
			DataTable dtTest = BOL.Supply.GetEmptySupplyTable();

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetEmptySupplyTable_Fail()
		{
		}

		[Test]
		public void GetManufacturerNameList_Pass()
		{
			System.Collections.ArrayList alTest = BOL.Supply.GetManufacturerNameList();
			
			Assert.AreEqual(true, alTest.Count >= 0, "Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetManufacturerNameList_Fail()
		{
		}

		
		[Test]
		public void GetSupplyNameList_Pass()
		{
			System.Collections.ArrayList alTest = BOL.Supply.GetSupplyNameList();
			
			Assert.AreEqual(true, alTest.Count >= 0, "Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetSupplyNameList_Fail()
		{
		}

		[Test]
		public void GetSupplyNameList_Manufacturer_Pass()
		{
			System.Collections.ArrayList alTest = BOL.Supply.GetSupplyNameList(this._dtSupply.Rows[0][TABLES.Supply.ManufacturerName].ToString());
			
			Assert.AreEqual(true, alTest.Count >= 0, "Count");
		}

		[Test]
		public void GetSupplyNameList_Manufacturer_Fail()
		{
			System.Collections.ArrayList alTest = BOL.Supply.GetSupplyNameList(Guid.NewGuid().ToString());
			
			Assert.AreEqual(0, alTest.Count, "Count");
		}

		[Test]
		public void GetSupplyTypeList_Pass()
		{
			System.Collections.ArrayList[] alTest = BOL.Supply.GetSupplyTypeList();
			
			Assert.AreEqual(true, alTest.Length > 0, "ArrayLength");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetSupplyTypeList_Fail()
		{
		}
		
		[Test]
		public void InsertSupplies_Pass()
		{
			this.RefreshData = true;

			DataRow dr = this._dtSupply.Rows[0];
			dr[TABLES.Supply.SupplyGuid] = System.Guid.NewGuid();

			BOL.Supply tmpSupply = new BOL.Supply(dr);
			System.Collections.ArrayList alTest = new System.Collections.ArrayList();

			//Make some changes
			tmpSupply.SupplyName = "Test" + new Random().Next(400,500).ToString();
			tmpSupply.InvoiceNumber = new Random().Next(1111111,9999999).ToString();
			tmpSupply.LotQuantity = new Random().Next(400,999);
			tmpSupply.AvailableQuantity = new Random().Next(100, tmpSupply.LotQuantity);
            
			
			alTest.Add(tmpSupply);

			bool retValue = BOL.Supply.InsertSupplies(alTest, Common.UpdateFunction.UnitTests);

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void InsertSupplies_Fail()
		{
		}
		
		[Test]
		public void Update_Pass()
		{
			this.RefreshData = true;

			BOL.Supply tmpSupply = new BOL.Supply(this._tstSupply.SupplyGuid);

			int updateQty = new Random().Next(0,tmpSupply.LotQuantity);

			tmpSupply.AvailableQuantity = updateQty;
			tmpSupply.InventoryChangeDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			tmpSupply.Comments = System.DateTime.Now.ToString();

			bool retValue = tmpSupply.Update(Common.UpdateFunction.UnitTests);

			Assert.IsTrue(retValue);

			tmpSupply = new BOL.Supply(this._tstSupply.SupplyGuid);

			Assert.AreEqual(updateQty, tmpSupply.AvailableQuantity, "Qty-Check");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Update_Fail()
		{
		}

		[Test]
		public void GetSupplyListSansContainerSCD_Pass()
		{
			ArrayList list = BOL.Supply.GetSupplyListSansContainerSCD( true );
			
			Assert.AreEqual(true, list.Count > 0,  "Count" );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetSupplyListSansContainerSCD_Fail()
		{
		}
	}
}
#endif


